/* eslint-disable @typescript-eslint/no-explicit-any */
import { fireEvent, screen } from '@testing-library/vue';
import FreeAiCreditsCallout from '@/app/components/FreeAiCreditsCallout.vue';
import { useCredentialsStore } from '@/features/credentials/credentials.store';
import { useSettingsStore } from '@/app/stores/settings.store';
import { useUsersStore } from '@/features/settings/users/users.store';
import { useNDVStore } from '@/features/ndv/shared/ndv.store';
import { useProjectsStore } from '@/features/collaboration/projects/projects.store';
import { useRootStore } from '@n8n/stores/useRootStore';
import { useToast } from '@/app/composables/useToast';
import { renderComponent } from '@/__tests__/render';
import { mockedStore } from '@/__tests__/utils';
import { useTelemetry } from '@/app/composables/useTelemetry';

vi.mock('@/app/composables/useToast', () => ({
	useToast: vi.fn(),
}));

vi.mock('@/app/composables/useTelemetry', () => ({
	useTelemetry: vi.fn(),
}));

vi.mock('@/app/stores/settings.store', () => ({
	useSettingsStore: vi.fn(),
}));

vi.mock('@/features/credentials/credentials.store', () => ({
	useCredentialsStore: vi.fn(),
}));

vi.mock('@/features/settings/users/users.store', () => ({
	useUsersStore: vi.fn(),
}));

vi.mock('@/features/ndv/shared/ndv.store', () => ({
	useNDVStore: vi.fn(),
}));

vi.mock('@/features/collaboration/projects/projects.store', () => ({
	useProjectsStore: vi.fn(),
}));

vi.mock('@n8n/stores/useRootStore', () => ({
	useRootStore: vi.fn(),
}));

const assertUserCannotClaimCredits = () => {
	expect(screen.queryByText('Get 100 free OpenAI API credits')).not.toBeInTheDocument();
	expect(screen.queryByRole('button', { name: 'Claim credits' })).not.toBeInTheDocument();
};

const assertUserCanClaimCredits = () => {
	expect(screen.getByText('Get 100 free OpenAI API credits')).toBeInTheDocument();
	expect(screen.queryByRole('button', { name: 'Claim credits' })).toBeInTheDocument();
};

const assertUserClaimedCredits = () => {
	expect(
		screen.getByText(
			'Claimed 100 free OpenAI API credits! Please note these free credits are only for the following models:',
		),
	).toBeInTheDocument();

	expect(
		screen.getByText(
			'gpt-4o-mini, text-embedding-3-small, dall-e-3, tts-1, whisper-1, and text-moderation-latest',
		),
	).toBeInTheDocument();
};

describe('FreeAiCreditsCallout', () => {
	beforeEach(() => {
		vi.clearAllMocks();

		(useSettingsStore as any).mockReturnValue({
			isAiCreditsEnabled: true,
			aiCreditsQuota: 100,
		});

		(useCredentialsStore as any).mockReturnValue({
			allCredentials: [],
			upsertCredential: vi.fn(),
			claimFreeAiCredits: vi.fn(),
		});

		(useUsersStore as any).mockReturnValue({
			currentUser: {
				settings: {
					userClaimedAiCredits: false,
				},
			},
		});

		(useNDVStore as any).mockReturnValue({
			activeNode: { type: '@n8n/n8n-nodes-langchain.openAi' },
		});

		(useProjectsStore as any).mockReturnValue({
			currentProject: { id: 'test-project-id' },
		});

		(useRootStore as any).mockReturnValue({
			restApiContext: {},
		});

		(useToast as any).mockReturnValue({
			showError: vi.fn(),
		});

		(useTelemetry as any).mockReturnValue({
			track: vi.fn(),
		});
	});

	it('should shows the claim callout when the user can claim credits', () => {
		renderComponent(FreeAiCreditsCallout);

		assertUserCanClaimCredits();
	});

	it('should show success callout when credit are claimed', async () => {
		const credentialsStore = mockedStore(useCredentialsStore);

		renderComponent(FreeAiCreditsCallout);

		const claimButton = screen.getByRole('button', {
			name: 'Claim credits',
		});

		await fireEvent.click(claimButton);

		expect(credentialsStore.claimFreeAiCredits).toHaveBeenCalledWith('test-project-id');
		expect(useTelemetry().track).toHaveBeenCalledWith('User claimed OpenAI credits');
		assertUserClaimedCredits();
	});

	it('should not be able to claim credits is user already claimed credits', async () => {
		(useUsersStore as any).mockReturnValue({
			currentUser: {
				settings: {
					userClaimedAiCredits: true,
				},
			},
		});

		renderComponent(FreeAiCreditsCallout);

		assertUserCannotClaimCredits();
	});

	it('should not be able to claim credits is user does not have ai credits enabled', async () => {
		(useSettingsStore as any).mockReturnValue({
			isAiCreditsEnabled: false,
			aiCreditsQuota: 0,
		});

		renderComponent(FreeAiCreditsCallout);

		assertUserCannotClaimCredits();
	});

	it('should not be able to claim credits if user already has OpenAiApi credential', async () => {
		(useCredentialsStore as any).mockReturnValue({
			allCredentials: [
				{
					type: 'openAiApi',
				},
			],
			upsertCredential: vi.fn(),
		});

		renderComponent(FreeAiCreditsCallout);

		assertUserCannotClaimCredits();
	});

	it('should not be able to claim credits if active node it is not a valid node', async () => {
		(useNDVStore as any).mockReturnValue({
			activeNode: { type: '@n8n/n8n-nodes.jira' },
		});

		renderComponent(FreeAiCreditsCallout);

		assertUserCannotClaimCredits();
	});

	describe('credentialTypeName prop (credentials page)', () => {
		it('should not show claim callout when editing a non-openapi credential', async () => {
			(useNDVStore as any).mockReturnValue({
				activeNode: null,
			});

			renderComponent(FreeAiCreditsCallout, {
				props: {
					credentialTypeName: 'googleSheetsOAuth2Api',
				},
			});

			assertUserCannotClaimCredits();
		});

		it('should show claim callout editing openapi credential with no active node', async () => {
			(useNDVStore as any).mockReturnValue({
				activeNode: null,
			});

			renderComponent(FreeAiCreditsCallout, {
				props: {
					credentialTypeName: 'openAiApi',
				},
			});

			assertUserCanClaimCredits();
		});

		it('should not show claim callout when credential type is undefined and no valid active node', async () => {
			(useNDVStore as any).mockReturnValue({
				activeNode: null,
			});

			renderComponent(FreeAiCreditsCallout, {
				props: {
					credentialTypeName: undefined,
				},
			});

			assertUserCannotClaimCredits();
		});
	});
});
