import type {
	IWorkflowDb,
	NewWorkflowResponse,
	WorkflowListResource,
	WorkflowResource,
} from '@/Interface';
import type {
	IExecutionResponse,
	IExecutionsCurrentSummaryExtended,
} from '@/features/execution/executions/executions.types';
import type { IRestApiContext } from '@n8n/rest-api-client';
import type {
	ExecutionFilters,
	ExecutionOptions,
	ExecutionSummary,
	IDataObject,
} from 'n8n-workflow';
import { getFullApiResponse, makeRestApiRequest } from '@n8n/rest-api-client';

export async function getNewWorkflow(context: IRestApiContext, data?: IDataObject) {
	const response = await makeRestApiRequest<NewWorkflowResponse>(
		context,
		'GET',
		'/workflows/new',
		data,
	);
	return {
		name: response.name,
		settings: response.defaultSettings,
	};
}

export async function getWorkflow(context: IRestApiContext, id: string) {
	return await makeRestApiRequest<IWorkflowDb>(context, 'GET', `/workflows/${id}`);
}

export async function workflowExists(context: IRestApiContext, id: string) {
	return await makeRestApiRequest<{ exists: boolean }>(context, 'GET', `/workflows/${id}/exists`);
}

export async function getWorkflows(
	context: IRestApiContext,
	filter?: object,
	options?: object,
	select?: string[],
) {
	return await getFullApiResponse<IWorkflowDb[]>(context, 'GET', '/workflows', {
		includeScopes: true,
		...(filter ? { filter } : {}),
		...(options ? options : {}),
		...(select ? { select: JSON.stringify(select) } : {}),
	});
}

export async function getWorkflowsWithNodesIncluded(context: IRestApiContext, nodeTypes: string[]) {
	return await getFullApiResponse<WorkflowResource[]>(
		context,
		'POST',
		'/workflows/with-node-types',
		{
			nodeTypes,
		},
	);
}

export async function getWorkflowsAndFolders(
	context: IRestApiContext,
	filter?: object,
	options?: object,
	includeFolders?: boolean,
	onlySharedWithMe?: boolean,
) {
	return await getFullApiResponse<WorkflowListResource[]>(context, 'GET', '/workflows', {
		includeScopes: true,
		includeFolders,
		onlySharedWithMe,
		...(filter ? { filter } : {}),
		...(options ? options : {}),
	});
}

export async function getActiveWorkflows(context: IRestApiContext) {
	return await makeRestApiRequest<string[]>(context, 'GET', '/active-workflows');
}

export async function getActiveExecutions(context: IRestApiContext, filter: IDataObject) {
	const output = await makeRestApiRequest<{
		results: ExecutionSummary[];
		count: number;
		estimated: boolean;
	}>(context, 'GET', '/executions', { filter });

	return output.results;
}

export async function getExecutions(
	context: IRestApiContext,
	filter?: ExecutionFilters,
	options?: ExecutionOptions,
): Promise<{ count: number; results: IExecutionsCurrentSummaryExtended[]; estimated: boolean }> {
	return await makeRestApiRequest(context, 'GET', '/executions', { filter, ...options });
}

export async function getExecutionData(context: IRestApiContext, executionId: string) {
	return await makeRestApiRequest<IExecutionResponse | null>(
		context,
		'GET',
		`/executions/${executionId}`,
	);
}

export async function getLastSuccessfulExecution(
	context: IRestApiContext,
	workflowId: string,
): Promise<IExecutionResponse | null> {
	return await makeRestApiRequest<IExecutionResponse | null>(
		context,
		'GET',
		`/workflows/${workflowId}/executions/last-successful`,
	);
}

export async function activateWorkflow(
	context: IRestApiContext,
	workflowId: string,
	data: { versionId: string; name?: string; description?: string },
): Promise<IWorkflowDb> {
	return await makeRestApiRequest<IWorkflowDb>(
		context,
		'POST',
		`/workflows/${workflowId}/activate`,
		data,
	);
}

export async function deactivateWorkflow(
	context: IRestApiContext,
	workflowId: string,
): Promise<IWorkflowDb> {
	return await makeRestApiRequest<IWorkflowDb>(
		context,
		'POST',
		`/workflows/${workflowId}/deactivate`,
	);
}
