import type { INodeParameters } from 'n8n-workflow';
import type { ZodIssue } from 'zod';

import type { PromptCategorization } from './categorization';
import type { AddedNode, NodeDetails, NodeSearchResult } from './nodes';
import type { SimpleWorkflow } from './workflow';

/**
 * Types of progress updates
 */
export type ProgressUpdateType = 'input' | 'output' | 'progress' | 'error';

/**
 * Progress update during tool execution
 */
export interface ProgressUpdate<T = Record<string, unknown>> {
	type: ProgressUpdateType;
	data: T;
	timestamp?: string;
}

/**
 * Tool progress message for streaming updates
 */
export interface ToolProgressMessage<TToolName extends string = string> {
	type: 'tool';
	toolName: TToolName;
	toolCallId?: string;
	status: 'running' | 'completed' | 'error';
	updates: ProgressUpdate[];
	displayTitle?: string; // Name of tool action in UI, for example "Adding nodes"
	customDisplayTitle?: string; // Custom name for tool action in UI, for example "Adding Gmail node"
}

/**
 * Tool execution error
 */
export interface ToolError {
	message: string;
	code?: string;
	details?: ZodIssue[] | Record<string, unknown>;
}

/**
 * Progress reporter interface for tools
 */
export interface ProgressReporter {
	start: <T>(input: T, options?: { customDisplayTitle: string }) => void;
	progress: (message: string, data?: Record<string, unknown>) => void;
	complete: <T>(output: T) => void;
	error: (error: ToolError) => void;
	createBatchReporter: (scope: string) => BatchReporter;
}

/**
 * Batch progress reporter for multi-item operations
 */
export interface BatchReporter {
	init: (total: number) => void;
	next: (itemDescription: string) => void;
	complete: () => void;
}

/**
 * Output type for update node parameters tool
 */
export interface UpdateNodeParametersOutput {
	nodeId: string;
	nodeName: string;
	nodeType: string;
	updatedParameters: INodeParameters;
	appliedChanges: string[];
	message: string;
}

/**
 * Output type for add node tool
 */
export interface AddNodeOutput {
	addedNode: AddedNode;
	message: string;
}

/**
 * Output type for connect nodes tool
 */
export interface ConnectNodesOutput {
	sourceNode: string;
	targetNode: string;
	connectionType: string;
	swapped: boolean;
	message: string;
	found: {
		sourceNode: boolean;
		targetNode: boolean;
	};
}

/**
 * Output type for remove node tool
 */
export interface RemoveNodeOutput {
	removedNodeId: string;
	removedNodeName: string;
	removedNodeType: string;
	connectionsRemoved: number;
	message: string;
}

/**
 * Output type for node details tool
 */
export interface NodeDetailsOutput {
	details: NodeDetails;
	found: boolean;
	message: string;
}

/**
 * Output type for node search tool
 */
export interface NodeSearchOutput {
	results: Array<{
		query: string;
		results: NodeSearchResult[];
	}>;
	totalResults: number;
	message: string;
}

/**
 * Output type for get node parameter tool
 */
export interface GetNodeParameterOutput {
	message: string; // This is only to report success or error, without actual value (we don't need to send it to the frontend)
}

/**
 * Output type for remove connection tool
 */
export interface RemoveConnectionOutput {
	sourceNode: string;
	targetNode: string;
	connectionType: string;
	sourceOutputIndex: number;
	targetInputIndex: number;
	message: string;
}

/**
 * Output type for categorize prompt tool
 */
export interface CategorizePromptOutput {
	categorization: PromptCategorization;
}

/**
 * Description of a workflow example we have found
 */
export interface WorkflowMetadata {
	templateId: number;
	name: string;
	description?: string;
	workflow: SimpleWorkflow;
}

/**
 * A node configuration entry with version information
 */
export interface NodeConfigurationEntry {
	version: number;
	parameters: INodeParameters;
}

/**
 * Map of node type to array of parameter configurations with version info
 * Key: node type (e.g., 'n8n-nodes-base.telegram')
 * Value: array of configuration entries with version and parameters
 */
export type NodeConfigurationsMap = Record<string, NodeConfigurationEntry[]>;

/**
 * Output type for get workflow examples tool
 */
export interface GetWorkflowExamplesOutput {
	examples: Array<{
		name: string;
		description?: string;
		workflow: string;
	}>;
	totalResults: number;
}

/**
 * Output type for get node configuration examples tool
 */
export interface GetNodeConfigurationExamplesOutput {
	nodeType: string;
	totalFound: number;
	message: string;
}
